﻿##############################################################################################################
# 
# Name        : OptimizeNICforPVS.ps1
# Author      : Ingmar Verheij - http://www.ingmarverheij.com
# Version     : 1.0, 17 july 2013
#               - Initial release
#
#               1.1, 18 july 2013 
#				- Removes IPv6 binding instead of disabling IPv6 completely(using nvspbind)
#				- Could not remove NIC on 32-bit machine (added devcon x86 and x64)
#
#               1.2, 21 july 2013
#               - Ensures the script runs with elevated priviliges
#				- Removes "Unblock file" dialog for subprocesses
#               - Adds the script invocation path for dependent processes (requried for PowerShell ISE)
#				- Detects if IPv6 binding was already removed
#               - Iterates through NIC 0 ~ 255 to deal with removed NIC's (causing the script stop processing other NICs)
#
#               1.3, 23 july 2013
#               - Included NIC Intel 21140 (Hyper-V legacy NIC) just for visual reasons (no TCP offload features or other optimization)
#               - Added message to reboot machine
#               - Removed network- / MAC- address from NIC properties 
#               - Use Unblock-File cmdlet if PowerShell 3 is used
#
#               1.4, 13 october 2013
#               - Checks if registry key exists, skips if it doesn't		
#               - Improve UDP performance for vSphere 5.1 (FastSendDatagramThreshold)
#               - Set multiplication factor to the default UDP scavenge value (MaxEndpointCountMult)
#				- Disables IPv6 completely (DisabledComponents)
#				- Hide PVS status icon
#               - Hide VMware tools icon
#				- Disable VMware debug driver
#				- Disable Receive Side Scaling (RSS)
#				- ARP Cache lifespan increased
#
#               1.5, 18 march 2014
#				- Removed syntax error in if statement on line #285 - https://twitter.com/jdestombes/status/443792370489122816
#
#               1.6, 9 april 2014
#				- When the bnistack service (Citrix Network Stack Interface Driver) is started NICs are never removed
#				  (It is reported that NICs are removed when the script was run in private mode, after the image was captured)
#
#				1.7, 11 jun 2014
#				- Added "Citrix PV Network Adapter", a paravirtualized NIC in XenServer
#
# Description : Optimizes the network (adapters) for streaming disks with Citrix PVS
#               - Disables TCP/IP task offloading
#               - Removes IPv6 binding 
#               - Disabled TSO and DMA on XenTools
#               - Removes hidden NIC's (removed NIC's)
#                 * only Ethernet adapters (ifType = ethernetCsmacd(6) - http://www.iana.org/assignments/ianaiftype-mib/ianaiftype-mib)
#                 * root devices are excluded (for instance "WAN Miniport*")
#                 * software defined network adapters are excluded (for instance "RAS Async Adapter")
#               - Sets advanced NIC properties for E1000 (VMware), vmxnet3 (VMware), Xennet6 (XenServer), VMbus (Hyper-V) and Intel 21140 (Hyper-V) to optimize network traffic
# 
# Dependencies : devcon   - http://www.microsoft.com/en-us/download/details.aspx?id=11800) (make sure you get version 6.1.7600.16385 for x64)
#                nvspbind - http://archive.msdn.microsoft.com/nvspbind
#
# Usage        : The script runs without parameter but requires elevated privileges, this is enforced by the script.
#                
#                Features can be enabled / disabled in the global options section (eg. disabling tcp/ip task offloading, unbinding IPv6, etc.)
#                For the feature "optimize NICs" you can set additional advanced properties which are not enabled by default.
#                This is done in three steps
#                1) locate the corrrect NIC type after the "switch" function
#                2) Uncomment the statement "Set-ItemProperty" for the property you want to change
#                3) Change the value in the "Set-ItemProperty" (at the end of the line) to your desired value
#
#                Before you set the advanced property in the script it's recommended to test this first to understand the implications.
#                
##############################################################################################################



# ---------------------------- Global options -----------------------------------
$blnDisableTaskOffload = $true      			# Disables TCP/IP task offloading 								- http://support.citrix.com/article/CTX117491
$blnDisableTSODMAXenTools = $true 				# Disable TSO and DMA on XenTools 								- http://support.citrix.com/article/CTX125157
$blnIncreaseFastSendDatagramThreshold  = $true 	# Increases the UDP packet size to 1500 bytes  					- http://kb.vmware.com/selfservice/microsites/search.do?language=en_US&cmd=displayKC&externalId=2040065
$lbnSetMaxEndpointCountMult = $true				# Set multiplication factor to the default UDP scavenge value 	- http://support.microsoft.com/kb/2685007/en-us
$blnDisableIPv6 = $true							# Disables IPv6 completely 										- http://social.technet.microsoft.com/wiki/contents/articles/10130.root-causes-for-slow-boots-and-logons-sbsl.aspx
$blnHidePVSstatusicon = $true					# Hide PVS status icon 											- http://forums.citrix.com/thread.jspa?threadID=273278
$blnHideVMwareToolsIcon = $true					# Hide VMware tools icon 										- http://www.blackmanticore.com/c84d61182fa782af965e0abf2d82d6e6
$blnDisableVMwareDebugDriver = $true			# Disable VMware debug driver 									- http://kb.vmware.com/selfservice/microsites/search.do?language=en_US&cmd=displayKC&externalId=1007652
$blnDisableRSS = $true							# Disable Receive Side Scaling (RSS)							- http://kb.vmware.com/selfservice/microsites/search.do?language=en_US&cmd=displayKC&externalId=2055853 // http://support.microsoft.com/kb/951037/en-us
$blnIncreaseARPCacheLifespan = $true			# Increases the ARP cache life span 							- http://support.citrix.com/article/ctx127549
$blnRemoveIPv6Binding = $true					# Removes IPv6 binding from network adapters
$blnRemoveHiddenNICs = $true					# Remove hidden NIC's
$blnOptimizeNICs = $true						# Optimize ethernet adapters


$stPathDevConx86 = (Split-Path $MyInvocation.MyCommand.Path) + "\devcon\devcon32.exe"     	 # Path to 32-bit DevCon
$stPathDevConx64 = (Split-Path $MyInvocation.MyCommand.Path) + "\devcon\devcon64.exe"     	 # Path to 64-bit DevCon
$strPathnvspbindx86 = (Split-Path $MyInvocation.MyCommand.Path) + "\nvspbind\nvspbind32.exe" # Path to 32-bit "Hyper-V Network VSP Bind Application"
$strPathnvspbindx64 = (Split-Path $MyInvocation.MyCommand.Path) + "\nvspbind\nvspbind64.exe" # Path to 64-bit "Hyper-V Network VSP Bind Application"
# -------------------------------------------------------------------------------


# ===============================================================================
# ==== Only change values below this line if you understand the implications ====
# ===============================================================================


# ------------------------------ Functions --------------------------------------
function Use-RunAs {    
    # Check if script is running as Adminstrator and if not use RunAs 
    # Use Check Switch to check if admin 
    # http://gallery.technet.microsoft.com/scriptcenter/63fd1c0d-da57-4fb4-9645-ea52fc4f1dfb
    
    param([Switch]$Check) 
    $IsAdmin = ([Security.Principal.WindowsPrincipal] [Security.Principal.WindowsIdentity]::GetCurrent()).IsInRole([Security.Principal.WindowsBuiltInRole] "Administrator") 
    if ($Check) { return $IsAdmin }     
    if ($MyInvocation.ScriptName -ne "") 
    {  
        if (-not $IsAdmin)  
        {  
            try 
            {  
                $arg = "-file `"$($MyInvocation.ScriptName)`"" 
                Start-Process "$psHome\powershell.exe" -Verb Runas -ArgumentList $arg -ErrorAction 'stop'  
            } 
            catch 
            { 
                Write-Warning "Error - Failed to restart script with runas"  
                break               
            } 
            exit # Quit this session of powershell 
        }  
    }  
    else  
    {  
        Write-Warning "Error - Script must be saved as a .ps1 file first"  
        break  
    }  
}
function Unblock-Files {
	
	# http://andyarismendi.blogspot.nl/2012/02/unblocking-files-with-powershell.html
	[cmdletbinding(DefaultParameterSetName="ByName", SupportsShouldProcess=$True)]
	param (
	       [parameter(Mandatory=$true, ParameterSetName="ByName", Position=0)] [string] $FilePath,
	       [parameter(Mandatory=$true, ParameterSetName="ByInput", ValueFromPipeline=$true)] $InputObject
  	      )
	begin 
	{       
		Add-Type -Namespace Win32 -Name PInvoke -MemberDefinition @"
		// http://msdn.microsoft.com/en-us/library/windows/desktop/aa363915(v=vs.85).aspx
		[DllImport("kernel32", CharSet = CharSet.Unicode, SetLastError = true)]
		[return: MarshalAs(UnmanagedType.Bool)]
		private static extern bool DeleteFile(string name);
		public static int Win32DeleteFile(string filePath) {
			bool is_gone = DeleteFile(filePath); return Marshal.GetLastWin32Error();}
			
		[DllImport("kernel32.dll", CharSet = CharSet.Auto, SetLastError = true)]
		static extern int GetFileAttributes(string lpFileName);
		public static bool Win32FileExists(string filePath) {return GetFileAttributes(filePath) != -1;}
"@
	}
	process 
	{
	
		switch ($PSCmdlet.ParameterSetName) 
		{
			'ByName' {$input_paths = Resolve-Path -Path $FilePath | ? {[IO.File]::Exists($_.Path)} | Select -Exp Path}
			'ByInput' {if ($InputObject -is [System.IO.FileInfo]) {$input_paths = $InputObject.FullName}}
		}
		
		$input_paths | % { 
			#Determine PowerShell engine version
			if ($PSVersionTable.psversion.Major -ge 3) {
				
				#PowerShell 3 has a native cmdlet
				Unblock-File -Path $_
			}
			else 
			{ 
				#PowerShell 1/2: Use pinvoke to access kernel32.dll API's 
				if ([Win32.PInvoke]::Win32FileExists($_ + ':Zone.Identifier')) 
				{
					if ($PSCmdlet.ShouldProcess($_)) 
					{
						$result_code = [Win32.PInvoke]::Win32DeleteFile($_ + ':Zone.Identifier')
						if ([Win32.PInvoke]::Win32FileExists($_ + ':Zone.Identifier')) {
						Write-Error ("Failed to unblock '{0}' the Win32 return code is '{1}'." -f $_, $result_code)
					}
					}
				}
			}
		}
	}
}
# -------------------------------------------------------------------------------


# Ensure the script runs with elevated priviliges
Use-RunAs
# -


# Start log transcript
Start-Transcript -Path ($MyInvocation.MyCommand.Definition -replace 'ps1','log') -Append | out-null
# -


# Remove unblock dialog for subprocesses
Unblock-Files -Filepath $stPathDevConx86 
Unblock-Files -Filepath $stPathDevConx64 
Unblock-Files -Filepath $strPathnvspbindx86 
Unblock-Files -Filepath $strPathnvspbindx64 
# -


# Disable TCP/IP task offloading 
# http://support.citrix.com/article/CTX117491
if ($blnDisableTaskOffload)
{
	#Inform user
	Write-Host -NoNewline -ForegroundColor White "Disabling TCP/IP task offloading"
	$strRegPath = "HKLM:\SYSTEM\CurrentControlSet\Services\Tcpip\Parameters"
	If (Test-Path -Path $strRegPath)
	{ Set-ItemProperty -Path $strRegPath -Name "DisableTaskOffload" -Value "1" -Type DWORD 
	  Write-Host -ForegroundColor Green " (done)" 
	} else { Write-Host -ForegroundColor Yellow " (skipped)"} 	
}
# -


# Disable TSO and DMA on XenTools
# http://support.citrix.com/article/CTX125157
if ($blnDisableTSODMAXenTools)
{
	#Inform user
	Write-Host -NoNewline -ForegroundColor White "Disable TSO and DMA on XenTools"
	$strRegPath = "HKLM:\SYSTEM\CurrentControlSet\Services\xenevtchn"
	If (Test-Path -Path $strRegPath)
	{ Set-ItemProperty -Path $strRegKey -Name "SetFlags" -Value "0x30000" -Type DWORD 
	  Write-Host -ForegroundColor Green " (done)" 
	} else { Write-Host -ForegroundColor Yellow " (skipped)"} 	
}
# -


# Increases the UDP packet size to 1500 bytes for FastSend
# http://kb.vmware.com/selfservice/microsites/search.do?language=en_US&cmd=displayKC&externalId=2040065
if ($blnIncreaseFastSendDatagramThreshold)
{
	#Inform user
	Write-Host -NoNewline -ForegroundColor White "Increasing UDP FastSend threshold"
	$strRegPath = "HKLM:\SYSTEM\CurrentControlSet\Services\afd\Parameters"
	If (Test-Path -Path $strRegPath)
	{ Set-ItemProperty -Path $strRegPath -Name "FastSendDatagramThreshold" -Value "1500" -Type DWORD 
	  Write-Host -ForegroundColor Green " (done)" 
	} else { Write-Host -ForegroundColor Yellow " (skipped)"} 	   
}
# -


 # Set multiplication factor to the default UDP scavenge value (MaxEndpointCountMult)
# http://support.microsoft.com/kb/2685007/en-us
if ($lbnSetMaxEndpointCountMult)
{
	#Inform user
	Write-Host -NoNewline -ForegroundColor White "Set multiplication factor to the default UDP scavenge value"
	$strRegPath = "HKLM:\SYSTEM\CurrentControlSet\Services\BFE\Parameters"
	If (Test-Path -Path $strRegPath)
	{ Set-ItemProperty -Path $strRegPath -Name "MaxEndpointCountMult" -Value "0x10" -Type DWORD 
	  Write-Host -ForegroundColor Green " (done)" 
	} else { Write-Host -ForegroundColor Yellow " (skipped)"} 	
}
# -

# Disable IPv6 completely
# http://social.technet.microsoft.com/wiki/contents/articles/10130.root-causes-for-slow-boots-and-logons-sbsl.aspx
if ($blnDisableIPv6)
{
	#Inform user
	Write-Host -NoNewline -ForegroundColor White "Disable IPv6 completely"
	$strRegPath = "HKLM:\SYSTEM\CurrentControlSet\Services\TcpIp6\Parameters"
	If (Test-Path -Path $strRegPath)
	{ Set-ItemProperty -Path $strRegPath -Name "DisabledComponents" -Value "0xffffffff" -Type DWORD 
	  Write-Host -ForegroundColor Green " (done)" 
	} else { Write-Host -ForegroundColor Yellow " (skipped)"} 		
}
# -


# Hide PVS status icon 
# http://forums.citrix.com/thread.jspa?threadID=273278
if ($blnHidePVSstatusicon)
{
	#Inform user
	Write-Host -NoNewline -ForegroundColor White "Hide PVS status icon"
	$strRegPath = "HKLM:\SOFTWARE\CITRIX\ProvisioningServices"
	If (Test-Path -Path $strRegPath)
	{ Set-ItemProperty -Path $strRegPath -Name "StatusTray" -Value "0" -Type DWORD 
	  Write-Host -ForegroundColor Green " (done)" 
	} else { Write-Host -ForegroundColor Yellow " (skipped)"} 		
}
# -


# Hide VMware tools icon
# http://www.blackmanticore.com/c84d61182fa782af965e0abf2d82d6e6
if ($blnHideVMwareToolsIcon)
{
	#Inform user
	Write-Host -NoNewline -ForegroundColor White "Hide VMware tools icon"
	$strRegPath = "HKLM:\SOFTWARE\VMware, Inc.\VMware Tools"
	If (Test-Path -Path $strRegPath)
	{ Set-ItemProperty -Path $strRegPath -Name "ShowTray" -Value "0" -Type DWORD 
	  Write-Host -ForegroundColor Green " (done)" 
	} else { Write-Host -ForegroundColor Yellow " (skipped)"} 		
}
# -


# Disable VMware debug driver 
# http://kb.vmware.com/selfservice/microsites/search.do?language=en_US&cmd=displayKC&externalId=1007652
if ($blnDisableVMwareDebugDriver)
{
	#Inform user
	Write-Host -NoNewline -ForegroundColor White "Disable VMware debug driver"
	$strRegPath = "HKLM:\SYSTEM\CurrentControlSet\services\vmdebug"
	If (Test-Path -Path $strRegPath)
	{ Set-ItemProperty -Path $strRegKey  -Name "Start" -Value "4" -Type DWORD 
	  Write-Host -ForegroundColor Green " (done)" 
	} else { Write-Host -ForegroundColor Yellow " (skipped)"} 
}
# -


# Disable Receive Side Scaling (RSS)
# http://kb.vmware.com/selfservice/microsites/search.do?language=en_US&cmd=displayKC&externalId=2055853
# http://support.microsoft.com/kb/951037/en-us
if ($blnDisableRSS)
{
	#Inform user
	Write-Host -NoNewline -ForegroundColor White "Disable Receive Side Scaling (RSS)"
	Start-Process -FilePath "netsh.exe" -Argumentlist "int tcp set global rss=disable" -Wait -WindowStyle Hidden
	Write-Host -ForegroundColor Green " (done)"
}
# -


#Determine if the 'Citrix Network Stack Interface Driver' service is started
$blnRunningInvDisk = $false
Write-Host -NoNewLine -ForegroundColor White "The 'bnistack' service is "
$arrbnistackService = Get-Service bnistack -ErrorAction SilentlyContinue
If ($arrbnistackService.Length -eq 0) {
    Write-Host -ForegroundColor Yellow "NOT installed."
} else
{
    If ($arrbnistackService.Status -eq "Running") {
        
        #Inform user
        Write-Host -ForegroundColor Yellow "running."

        #The vDisk is streamed via PVS
        $blnRunningInvDisk = $true

	    #Inform user
	    Write-Host -NoNewline -ForegroundColor White "This vDisk is streamed via PVS in: "  
	
	    #Determine write cache type - https://support.citrix.com/article/CTX122224
	    [int]$intWriteCacheType = (Get-ItemProperty -Path "HKLM:\SYSTEM\CurrentControlSet\Services\bnistack\PvsAgent" -ErrorAction SilentlyContinue).WriteCacheType
	    [string]$strWriteCacheType = "(Unknown)"
	    switch ($intWriteCacheType)
	    {
		    0 {$strWriteCacheType = "Private Mode"}
		    1 {$strWriteCacheType = "Standarde Mode: Server Disk"}
		    2 {$strWriteCacheType = "Standarde Mode: Server Disk Encrypted"}
		    3 {$strWriteCacheType = "Standarde Mode: RAM"}
		    4 {$strWriteCacheType = "Standarde Mode: Hard Disk"}
		    5 {$strWriteCacheType = "Standarde Mode: Hard Disk Encrypted"}
		    6 {$strWriteCacheType = "Standarde Mode: RAM Disk"}
		    7 {$strWriteCacheType = "Standarde Mode: Difference Disk"}
	    }
	    Write-Host -ForegroundColor Yellow ("{0} [{1}]" -f  $strWriteCacheType, $intWriteCacheType) 

    } else {
        #Inform user
        Write-Host -ForegroundColor Yellow "installed."
	    Write-Host -ForegroundColor White "This disk is not streamed via PVS."

    }
}



#Inform user
Write-Host -ForegroundColor White "Iterating through network adapters"
$intNICid=0; do
{
	#Read network adapter properties
	$objNICproperties = (Get-ItemProperty -Path ("HKLM:\SYSTEM\CurrentControlSet\Control\Class\{0}\{1}" -f "{4D36E972-E325-11CE-BFC1-08002BE10318}", ( "{0:D4}" -f $intNICid)) -ErrorAction SilentlyContinue)
	
	#Determine if the Network adapter index exists 
	If ($objNICproperties)
	{
		#Filter network adapters
		# * only Ethernet adapters (ifType = ethernetCsmacd(6) - http://www.iana.org/assignments/ianaiftype-mib/ianaiftype-mib)
		# * root devices are exclude (for instance "WAN Miniport*")
		# * software defined network adapters are excluded (for instance "RAS Async Adapter")
		If (($objNICproperties."*ifType" -eq 6) -and 
		    ($objNICproperties.DeviceInstanceID -notlike "ROOT\*") -and
			($objNICproperties.DeviceInstanceID -notlike "SW\*")
			)
		{

			#Read hardware properties
			$objHardwareProperties = (Get-ItemProperty -Path ("HKLM:\SYSTEM\CurrentControlSet\Enum\{0}" -f $objNICproperties.DeviceInstanceID) -ErrorAction SilentlyContinue)
			If ($objHardwareProperties.FriendlyName)
			{ $strNICDisplayName = $objHardwareProperties.FriendlyName }
			else 
			{ $strNICDisplayName = $objNICproperties.DriverDesc }
			
			#Read Network properties
			$objNetworkProperties = (Get-ItemProperty -Path ("HKLM:\SYSTEM\CurrentControlSet\Control\Network\{0}\{1}\Connection" -f "{4D36E972-E325-11CE-BFC1-08002BE10318}", $objNICproperties.NetCfgInstanceId) -ErrorAction SilentlyContinue)
		      
            #Inform user
			Write-Host -NoNewline -ForegroundColor White "   ID     : "; Write-Host -ForegroundColor Yellow ( "{0:D4}" -f $intNICid)
			Write-Host -NoNewline -ForegroundColor White "   Network: "; Write-Host $objNetworkProperties.Name
            Write-Host -NoNewline -ForegroundColor White "   NIC    : "; Write-Host $strNICDisplayName
            Write-Host -ForegroundColor White "   Actions"
			
			#Determine if the vDisk is streamed via PVS
			If ($blnRunningInvDisk) {
				
				#NIC count is always 1
				[int]$intNICReferenceCount = 1
			} else {
				
				#Determine if the NIC is linked to an actual device
				[int]$intNICReferenceCount = (Get-ItemProperty -Path ("HKLM:\SYSTEM\CurrentControlSet\Control\DeviceClasses\{0}\##?#{1}#{0}\Control" -f "{ad498944-762f-11d0-8dcb-00c04fc3358c}", ($objNICproperties.DeviceInstanceID -replace '\\','#')) -ErrorAction SilentlyContinue).ReferenceCount		
			}

			#Determine it the NIC is active
			If ($intNICReferenceCount -eq 0) 
			
			{
				#NIC is not connected to this machine: remove that NIC!
				if ($blnRemoveHiddenNICs) 
				{
                    #Remove hardware with devcon - http://support.microsoft.com/kb/311272/en-us
					if ([System.IntPtr]::Size -eq 4) 
					{ Start-Process -FilePath $stPathDevConx86 -Argumentlist ('remove "@{0}"' -f $objNICproperties.DeviceInstanceID) -Wait -WindowStyle Hidden # 32-bit
					} else
					{ Start-Process -FilePath $stPathDevConx64 -Argumentlist ('remove "@{0}"' -f $objNICproperties.DeviceInstanceID) -Wait -WindowStyle Hidden # 32-bit
					}
					
					
                    Write-Host -ForegroundColor Red ("   - Removed")
				}
			} else {
			   
			   # Increases the ARP cache lifespan
			   # http://support.citrix.com/article/ctx127549
			   if ($blnIncreaseARPCacheLifespan)
			   {
			      #Determine InterfaceIndex
				  $interfaceIndex=(get-wmiobject win32_networkadapter -filter ("GUID = '{0}'" -f $objNICproperties.NetCfgInstanceId) -Property InterfaceIndex).InterfaceIndex
				  Start-Process -FilePath "netsh.exe" -Argumentlist ('interface ipv4 set interface {0} basereachable=600000' -f $interfaceIndex) -Wait -WindowStyle Hidden # 32-bit
				  
				  Write-Host -ForegroundColor Green ("   - ARP cache lifespan increased")
			   }	
			   
			   	#NIC is connected to this machine
				# - remove IPv6 binding 
				if ($blnRemoveIPv6Binding)
				{
				    #Determine if IPv6 is bind
					$objNIClinkage = (Get-ItemProperty -Path ("HKLM:\SYSTEM\CurrentControlSet\Control\Class\{0}\{1}\Linkage" -f "{4D36E972-E325-11CE-BFC1-08002BE10318}", ( "{0:D4}" -f $intNICid)) -ErrorAction SilentlyContinue)
					If( $objNIClinkage.UpperBind -contains "Tcpip6") 
					{
	                    #Remove IPv6 binding with nvspbind - http://archive.msdn.microsoft.com/nvspbind
						if ([System.IntPtr]::Size -eq 4) 
						{ Start-Process -FilePath $strPathnvspbindx86 -Argumentlist ('/d {0} ms_tcpip6' -f $objNICproperties.NetCfgInstanceId) -Wait -WindowStyle Hidden # 32-bit
						} else
						{ Start-Process -FilePath $strPathnvspbindx64 -Argumentlist ('/d {0} ms_tcpip6' -f $objNICproperties.NetCfgInstanceId) -Wait -WindowStyle Hidden # 64-bit 
						}
					
						Write-Host -ForegroundColor Green ("   - IPv6 binding removed")
					}
				}
				
				#- optimize that NIC!
				if ($blnOptimizeNICs) 
				{
				 	
					#Determine NIC type
					switch ($objNICproperties.DriverDesc.ToLower())
					{

						#E1000 (VMware)
						"Intel(R) PRO/1000 MT Network Connection".Tolower() 
						{
							# --- Flow Control ---
							$strRegistryKeyName = "*FlowControl" 
							# 0 - Disabled (default)
							# 1 - Tx Enabled
							# 2 - Rx Enabled
							# 3 - Tx and Rx Enabled
							#Set-ItemProperty -Path ("HKLM:\SYSTEM\CurrentControlSet\Control\Class\{0}\{1}" -f "{4D36E972-E325-11CE-BFC1-08002BE10318}", ( "{0:D4}" -f $intNICid)) -Name $strRegistryKeyName -Value "0"
							
							# --- Interrupt Moderation ---
							$strRegistryKeyName = "*InterruptModeration"
							# 0 - Disabled 
							# 1 - Enabled (default)
							#Set-ItemProperty -Path ("HKLM:\SYSTEM\CurrentControlSet\Control\Class\{0}\{1}" -f "{4D36E972-E325-11CE-BFC1-08002BE10318}", ( "{0:D4}" -f $intNICid)) -Name $strRegistryKeyName -Value "1"
							
							# --- IPv4 Checksum Offload ---
							$strRegistryKeyName = "*IPChecksumOffloadIPv4"
							# 0 - Disabled 
							# 1 - Tx Enabled
							# 2 - Rx Enabled
							# 3 - Tx and Rx Enabled (default)
							#Set-ItemProperty -Path ("HKLM:\SYSTEM\CurrentControlSet\Control\Class\{0}\{1}" -f "{4D36E972-E325-11CE-BFC1-08002BE10318}", ( "{0:D4}" -f $intNICid)) -Name $strRegistryKeyName -Value "3"

							# --- Jumbo Packet ---
							$strRegistryKeyName = "*JumboPacket"
							# 1514 - Disabled (Default)
							# 4088 - 4088 Bytes
							# 9014 - 9014 Bytes (Alteon)
							# 16128 - 16128 Bytes
							#Set-ItemProperty -Path ("HKLM:\SYSTEM\CurrentControlSet\Control\Class\{0}\{1}" -f "{4D36E972-E325-11CE-BFC1-08002BE10318}", ( "{0:D4}" -f $intNICid)) -Name $strRegistryKeyName -Value "1514"

							# --- Large Send Offload (IPv4) ---
							$strRegistryKeyName = "*LsoV1IPv4"
							# 0 - Disabled 
							# 1 - Enabled (default)
							Set-ItemProperty -Path ("HKLM:\SYSTEM\CurrentControlSet\Control\Class\{0}\{1}" -f "{4D36E972-E325-11CE-BFC1-08002BE10318}", ( "{0:D4}" -f $intNICid)) -Name $strRegistryKeyName -Value "0"
							
							# --- Priority & VLAN ---
							$strRegistryKeyName = "*PriorityVLANTag"
							# 0 - Priority & VLAN Disabled
							# 1 - Priority Enabled
							# 2 - VLAN Enabled
							# 3 - Priority & VLAN Enabled (default)
							#Set-ItemProperty -Path ("HKLM:\SYSTEM\CurrentControlSet\Control\Class\{0}\{1}" -f "{4D36E972-E325-11CE-BFC1-08002BE10318}", ( "{0:D4}" -f $intNICid)) -Name $strRegistryKeyName -Value "3"

							# --- Receive Buffers ---
							$strRegistryKeyName = "*ReceiveBuffers"
							# Value between 80 and 2048 with steps of 8, default is 256
							#Set-ItemProperty -Path ("HKLM:\SYSTEM\CurrentControlSet\Control\Class\{0}\{1}" -f "{4D36E972-E325-11CE-BFC1-08002BE10318}", ( "{0:D4}" -f $intNICid)) -Name $strRegistryKeyName -Value "256"

							# --- Link Speed & Duplex ---
							$strRegistryKeyName = "*SpeedDuplex"
							# 0 - Auto Negotiation (default)
							# 1 -  10 Mbps Half Duplex
							# 2 -  10 Mbps Full Duplex
							# 3 - 100 Mbps Half Duplex
							# 4 - 100 Mbps Full Duplex
							# 5 - 1000 Mbps Full Duplex
							#Set-ItemProperty -Path ("HKLM:\SYSTEM\CurrentControlSet\Control\Class\{0}\{1}" -f "{4D36E972-E325-11CE-BFC1-08002BE10318}", ( "{0:D4}" -f $intNICid)) -Name $strRegistryKeyName -Value "0"

							# --- TCP Checksum Offload (IPv4) ---
							$strRegistryKeyName = "*TCPChecksumOffloadIPv4"
							# 0 - Disabled 
							# 1 - Tx Enabled
							# 2 - Rx Enabled
							# 3 - Tx and Rx Enabled (default)
							#Set-ItemProperty -Path ("HKLM:\SYSTEM\CurrentControlSet\Control\Class\{0}\{1}" -f "{4D36E972-E325-11CE-BFC1-08002BE10318}", ( "{0:D4}" -f $intNICid)) -Name $strRegistryKeyName -Value "3"

							# --- Transmit Buffers ---
							$strRegistryKeyName = "*TransmitBuffers"
							# Value between 80 and 2048 with steps of 8, default is 512
							#Set-ItemProperty -Path ("HKLM:\SYSTEM\CurrentControlSet\Control\Class\{0}\{1}" -f "{4D36E972-E325-11CE-BFC1-08002BE10318}", ( "{0:D4}" -f $intNICid)) -Name $strRegistryKeyName -Value "512"

							# --- UDP Checksum Offload (IPv4) ---
							$strRegistryKeyName = "*UDPChecksumOffloadIPv4"
							# 0 - Disabled 
							# 1 - Tx Enabled
							# 2 - Rx Enabled
							# 3 - Tx and Rx Enabled (default)							
							#Set-ItemProperty -Path ("HKLM:\SYSTEM\CurrentControlSet\Control\Class\{0}\{1}" -f "{4D36E972-E325-11CE-BFC1-08002BE10318}", ( "{0:D4}" -f $intNICid)) -Name $strRegistryKeyName -Value "3"

							# --- Adaptive Inter-Frame Spacing ---
							$strRegistryKeyName = "AdaptiveIFS"
							# 0 - Disabled 
							# 1 - Enabled (default)
							#Set-ItemProperty -Path ("HKLM:\SYSTEM\CurrentControlSet\Control\Class\{0}\{1}" -f "{4D36E972-E325-11CE-BFC1-08002BE10318}", ( "{0:D4}" -f $intNICid)) -Name $strRegistryKeyName -Value "1"
							
							# --- Interrupt Moderation Rate ---
							$strRegistryKeyName = "ITR"
							# 65535 - Adaptive (default)
							# 3600 - Extreme
							# 2000 - High
							# 950 - Medium
							# 400 - Low
							# 200 - Minimal
							# 0 - Off
							#Set-ItemProperty -Path ("HKLM:\SYSTEM\CurrentControlSet\Control\Class\{0}\{1}" -f "{4D36E972-E325-11CE-BFC1-08002BE10318}", ( "{0:D4}" -f $intNICid)) -Name $strRegistryKeyName -Value "65535"

							# --- Number of Coalesce Buffers ---
							$strRegistryKeyName = "NumCoalesceBuffers"
							# Value between 16 and 768, default is 128
							#Set-ItemProperty -Path ("HKLM:\SYSTEM\CurrentControlSet\Control\Class\{0}\{1}" -f "{4D36E972-E325-11CE-BFC1-08002BE10318}", ( "{0:D4}" -f $intNICid)) -Name $strRegistryKeyName -Value "128"
                            
                            
                            Write-Host -ForegroundColor Green ("   - Optimized")
						}
						
						#vmxnet3 (VMware)
						"vmxnet3 Ethernet Adapter".Tolower() 
						{
							# --- Enable adaptive rx ring sizing ---
							$strRegistryKeyName = "*EnableAdaptiveRing"
							# 0 - Disabled 
							# 1 - Enabled (default)
							#Set-ItemProperty -Path ("HKLM:\SYSTEM\CurrentControlSet\Control\Class\{0}\{1}" -f "{4D36E972-E325-11CE-BFC1-08002BE10318}", ( "{0:D4}" -f $intNICid)) -Name $strRegistryKeyName -Value "1"
							
							# --- Interrupt Moderation ---
							$strRegistryKeyName = "*InterruptModeration"
							# 0 - Disabled 
							# 1 - Enabled (default)
							#Set-ItemProperty -Path ("HKLM:\SYSTEM\CurrentControlSet\Control\Class\{0}\{1}" -f "{4D36E972-E325-11CE-BFC1-08002BE10318}", ( "{0:D4}" -f $intNICid)) -Name $strRegistryKeyName -Value "1"

							# --- IPv4 Checksum Offload ---
							$strRegistryKeyName = "*IPChecksumOffloadIPv4"
							# 0 - Disabled
							# 1 - Tx Enabled
							# 2 - Rx Enabled
							# 3 - Tx and Rx Enabled (default)
							#Set-ItemProperty -Path ("HKLM:\SYSTEM\CurrentControlSet\Control\Class\{0}\{1}" -f "{4D36E972-E325-11CE-BFC1-08002BE10318}", ( "{0:D4}" -f $intNICid)) -Name $strRegistryKeyName -Value "3"

							# --- Jumbo Packet ---
							$strRegistryKeyName = "*JumboPacket"
							# 1514 - Standard 1500 (default)
							# 9014 - Jumbo 9000
							#Set-ItemProperty -Path ("HKLM:\SYSTEM\CurrentControlSet\Control\Class\{0}\{1}" -f "{4D36E972-E325-11CE-BFC1-08002BE10318}", ( "{0:D4}" -f $intNICid)) -Name $strRegistryKeyName -Value "1514"

							# --- IPv4 TSO Offload ---
							$strRegistryKeyName = "*LsoV1IPv4"
							# 0 - Disabled 
							# 1 - Enabled (default)
							Set-ItemProperty -Path ("HKLM:\SYSTEM\CurrentControlSet\Control\Class\{0}\{1}" -f "{4D36E972-E325-11CE-BFC1-08002BE10318}", ( "{0:D4}" -f $intNICid)) -Name $strRegistryKeyName -Value "0"
							
							# --- IPv4 Giant TSO Offload ---
							$strRegistryKeyName = "*LsoV2IPv4"
							# 0 - Disabled 
							# 1 - Enabled (default)
							Set-ItemProperty -Path ("HKLM:\SYSTEM\CurrentControlSet\Control\Class\{0}\{1}" -f "{4D36E972-E325-11CE-BFC1-08002BE10318}", ( "{0:D4}" -f $intNICid)) -Name $strRegistryKeyName -Value "0"

							# --- IPv6 TCP Segmentation Offload ---
							$strRegistryKeyName = "*LsoV2IPv6"
							# 0 - Disabled 
							# 1 - Enabled (default)
							#Set-ItemProperty -Path ("HKLM:\SYSTEM\CurrentControlSet\Control\Class\{0}\{1}" -f "{4D36E972-E325-11CE-BFC1-08002BE10318}", ( "{0:D4}" -f $intNICid)) -Name $strRegistryKeyName -Value "1"

							#--- Maximum number of RSS Processors ---
							$strRegistryKeyName = "*MaxRssProcessors"
							# 1 - 1
							# 2 - 2
							# 4 - 4
							# 8 - 8 (default)
							#Set-ItemProperty -Path ("HKLM:\SYSTEM\CurrentControlSet\Control\Class\{0}\{1}" -f "{4D36E972-E325-11CE-BFC1-08002BE10318}", ( "{0:D4}" -f $intNICid)) -Name $strRegistryKeyName -Value "8"

							# --- Priority / VLAN tag ---
							$strRegistryKeyName = "*PriorityVLANTag"
							# 0 - Priority & VLAN Disabled
							# 1 - Priority Enabled
							# 2 - VLAN Enabled (default)
							# 3 - Priority & VLAN Enabled
							#Set-ItemProperty -Path ("HKLM:\SYSTEM\CurrentControlSet\Control\Class\{0}\{1}" -f "{4D36E972-E325-11CE-BFC1-08002BE10318}", ( "{0:D4}" -f $intNICid)) -Name $strRegistryKeyName -Value "0"

							# --- RSS ---
							$strRegistryKeyName = "*RSS"
							# 0 - Disabled (default) 
							# 1 - Enabled 
							#Set-ItemProperty -Path ("HKLM:\SYSTEM\CurrentControlSet\Control\Class\{0}\{1}" -f "{4D36E972-E325-11CE-BFC1-08002BE10318}", ( "{0:D4}" -f $intNICid)) -Name $strRegistryKeyName -Value "0"

							# --- RSS Base Processor Number ---
							$strRegistryKeyName = "*RssBaseProcNumber"
							# value between 0 and 63, default is 63 
							#Set-ItemProperty -Path ("HKLM:\SYSTEM\CurrentControlSet\Control\Class\{0}\{1}" -f "{4D36E972-E325-11CE-BFC1-08002BE10318}", ( "{0:D4}" -f $intNICid)) -Name $strRegistryKeyName -Value "63"

							# --- Speed / Duplex ---
							$strRegistryKeyName = "*SpeedDuplex"
							# 0 - Auto Negotiation (default)
							# 1 -  10 Mbps Half Duplex
							# 2 -  10 Mbps Full Duplex
							# 3 - 100 Mbps Half Duplex
							# 4 - 100 Mbps Full Duplex
							# 5 - 1.0 Gbps Half Duplex
							# 6 - 1.0 Gbps Full Duplex
							# 7 -  10 Gbps Full Duplex
							#Set-ItemProperty -Path ("HKLM:\SYSTEM\CurrentControlSet\Control\Class\{0}\{1}" -f "{4D36E972-E325-11CE-BFC1-08002BE10318}", ( "{0:D4}" -f $intNICid)) -Name $strRegistryKeyName -Value "0"

							# --- TCP Checksum Offload (IPv4) ---
							$strRegistryKeyName = "*TCPChecksumOffloadIPv4"
							# 0 - Disabled
							# 1 - Tx Enabled
							# 2 - Rx Enabled
							# 3 - Tx and Rx Enabled (default)
							#Set-ItemProperty -Path ("HKLM:\SYSTEM\CurrentControlSet\Control\Class\{0}\{1}" -f "{4D36E972-E325-11CE-BFC1-08002BE10318}", ( "{0:D4}" -f $intNICid)) -Name $strRegistryKeyName -Value "3"

							# --- TCP Checksum Offload (IPv6) ---
							$strRegistryKeyName = "*TCPChecksumOffloadIPv6"
							# 0 - Disabled
							# 1 - Tx Enabled
							# 2 - Rx Enabled
							# 3 - Tx and Rx Enabled (default)
							#Set-ItemProperty -Path ("HKLM:\SYSTEM\CurrentControlSet\Control\Class\{0}\{1}" -f "{4D36E972-E325-11CE-BFC1-08002BE10318}", ( "{0:D4}" -f $intNICid)) -Name $strRegistryKeyName -Value "3"

							# --- UDP Checksum Offload (IPv4) ---
							$strRegistryKeyName = "*UDPChecksumOffloadIPv4"
							# 0 - Disabled
							# 1 - Tx Enabled
							# 2 - Rx Enabled
							# 3 - Tx and Rx Enabled (default)
							#Set-ItemProperty -Path ("HKLM:\SYSTEM\CurrentControlSet\Control\Class\{0}\{1}" -f "{4D36E972-E325-11CE-BFC1-08002BE10318}", ( "{0:D4}" -f $intNICid)) -Name $strRegistryKeyName -Value "3"

							# --- UDP Checksum Offload (IPv6) ---
							$strRegistryKeyName = "*UDPChecksumOffloadIPv6"
							# 0 - Disabled
							# 1 - Tx Enabled
							# 2 - Rx Enabled
							# 3 - Tx and Rx Enabled (default)
							#Set-ItemProperty -Path ("HKLM:\SYSTEM\CurrentControlSet\Control\Class\{0}\{1}" -f "{4D36E972-E325-11CE-BFC1-08002BE10318}", ( "{0:D4}" -f $intNICid)) -Name $strRegistryKeyName -Value "3"

							# --- Wake on magic packet ---
							$strRegistryKeyName = "*WakeOnMagicPacket"
							# 0 - Disabled 
							# 1 - Enabled (default)							
							#Set-ItemProperty -Path ("HKLM:\SYSTEM\CurrentControlSet\Control\Class\{0}\{1}" -f "{4D36E972-E325-11CE-BFC1-08002BE10318}", ( "{0:D4}" -f $intNICid)) -Name $strRegistryKeyName -Value "1"

							# --- Wake on pattern match ---
							$strRegistryKeyName = "*WakeOnPattern"
							# 0 - Disabled 
							# 1 - Enabled (default)	
							#Set-ItemProperty -Path ("HKLM:\SYSTEM\CurrentControlSet\Control\Class\{0}\{1}" -f "{4D36E972-E325-11CE-BFC1-08002BE10318}", ( "{0:D4}" -f $intNICid)) -Name $strRegistryKeyName -Value "1"

							# --- Wake-on-LAN ---
							$strRegistryKeyName = "EnableWakeOnLan"
							# 0 - Disabled 
							# 1 - Enabled (default)
							#Set-ItemProperty -Path ("HKLM:\SYSTEM\CurrentControlSet\Control\Class\{0}\{1}" -f "{4D36E972-E325-11CE-BFC1-08002BE10318}", ( "{0:D4}" -f $intNICid)) -Name $strRegistryKeyName -Value "1"

							# --- Rx Ring #1 Size ---
							$strRegistryKeyName = "MaxRxRing1Length"
							# 32 - 32
							# 64 - 64
							# 128 - 128
							# 256 - 256
							# 512 - 512 (default)
							# 630 - 630
							# 768 - 768
							# 896 - 896
							# 1024 - 1024
							# 2048 - 2048
							# 4096 - 4096
							#Set-ItemProperty -Path ("HKLM:\SYSTEM\CurrentControlSet\Control\Class\{0}\{1}" -f "{4D36E972-E325-11CE-BFC1-08002BE10318}", ( "{0:D4}" -f $intNICid)) -Name $strRegistryKeyName -Value "512"

							# --- Rx Ring #2 Size ---
							$strRegistryKeyName = "MaxRxRing2Length"
							# 32 - 32 (default)
							# 64 - 64
							# 128 - 128
							# 256 - 256
							# 512 - 512 
							# 1024 - 1024
							# 2048 - 2048
							# 4096 - 4096
							#Set-ItemProperty -Path ("HKLM:\SYSTEM\CurrentControlSet\Control\Class\{0}\{1}" -f "{4D36E972-E325-11CE-BFC1-08002BE10318}", ( "{0:D4}" -f $intNICid)) -Name $strRegistryKeyName -Value "32"

							# --- Max Tx Queues ---
							$strRegistryKeyName = "MaxTxQueues"
							# 1 - 1 (default)
							# 2 - 2
							# 4 - 4
							# 8 - 8
							#Set-ItemProperty -Path ("HKLM:\SYSTEM\CurrentControlSet\Control\Class\{0}\{1}" -f "{4D36E972-E325-11CE-BFC1-08002BE10318}", ( "{0:D4}" -f $intNICid)) -Name $strRegistryKeyName -Value "1"

							# --- Tx Ring Size ---
							$strRegistryKeyName = "MaxTxRingLength"
							# 32 - 32 
							# 64 - 64
							# 128 - 128
							# 256 - 256
							# 512 - 512 (default)
							# 1024 - 1024
							# 2048 - 2048
							# 4096 - 4096
							#Set-ItemProperty -Path ("HKLM:\SYSTEM\CurrentControlSet\Control\Class\{0}\{1}" -f "{4D36E972-E325-11CE-BFC1-08002BE10318}", ( "{0:D4}" -f $intNICid)) -Name $strRegistryKeyName -Value "512"

							# --- Large Rx Buffers ---
							$strRegistryKeyName = "NumRxBuffersLarge"
							# 64 - 64
							# 128 - 128
							# 256 - 256
							# 512 - 512 
							# 768 - 768 (default)
							# 1024 - 1024
							# 2048 - 2048
							# 3072 - 3072
							# 4096 - 4096							
							# 8192 - 8192
							#Set-ItemProperty -Path ("HKLM:\SYSTEM\CurrentControlSet\Control\Class\{0}\{1}" -f "{4D36E972-E325-11CE-BFC1-08002BE10318}", ( "{0:D4}" -f $intNICid)) -Name $strRegistryKeyName -Value "768"

							# --- Small Rx Buffers ---
							$strRegistryKeyName = "NumRxBuffersSmall"
							# 64 - 64
							# 128 - 128
							# 256 - 256
							# 512 - 512 
							# 768 - 768 
							# 1024 - 1024 (default)
							# 2048 - 2048
							# 3072 - 3072
							# 4096 - 4096							
							# 8192 - 8192
							#Set-ItemProperty -Path ("HKLM:\SYSTEM\CurrentControlSet\Control\Class\{0}\{1}" -f "{4D36E972-E325-11CE-BFC1-08002BE10318}", ( "{0:D4}" -f $intNICid)) -Name $strRegistryKeyName -Value "1024"

							# --- Offload IP Options ---
							$strRegistryKeyName = "OffloadIpOptions"
							# 0 - Disabled 
							# 1 - Enabled (default)
							#Set-ItemProperty -Path ("HKLM:\SYSTEM\CurrentControlSet\Control\Class\{0}\{1}" -f "{4D36E972-E325-11CE-BFC1-08002BE10318}", ( "{0:D4}" -f $intNICid)) -Name $strRegistryKeyName -Value "1"

							# --- Offload TCP Options ---
							$strRegistryKeyName = "OffloadTcpOptions"
							# 0 - Disabled 
							# 1 - Enabled (default)
							#Set-ItemProperty -Path ("HKLM:\SYSTEM\CurrentControlSet\Control\Class\{0}\{1}" -f "{4D36E972-E325-11CE-BFC1-08002BE10318}", ( "{0:D4}" -f $intNICid)) -Name $strRegistryKeyName -Value "1"

							# --- Offload tagged traffic ---
							$strRegistryKeyName = "OffloadVlanEncap"
							# 0 - Disabled 
							# 1 - Enabled (default)
							#Set-ItemProperty -Path ("HKLM:\SYSTEM\CurrentControlSet\Control\Class\{0}\{1}" -f "{4D36E972-E325-11CE-BFC1-08002BE10318}", ( "{0:D4}" -f $intNICid)) -Name $strRegistryKeyName -Value "1"

							# --- Receive Throttle ----
							$strRegistryKeyName = "RxThrottle"
							# Value between 0 and 5000 with steps of 10 - default is 0
							#Set-ItemProperty -Path ("HKLM:\SYSTEM\CurrentControlSet\Control\Class\{0}\{1}" -f "{4D36E972-E325-11CE-BFC1-08002BE10318}", ( "{0:D4}" -f $intNICid)) -Name $strRegistryKeyName -Value "0"
							
							# --- VLAN ID ---
							$strRegistryKeyName = "VlanId"
							# Value between 0 and 4095 - default is 0
							#Set-ItemProperty -Path ("HKLM:\SYSTEM\CurrentControlSet\Control\Class\{0}\{1}" -f "{4D36E972-E325-11CE-BFC1-08002BE10318}", ( "{0:D4}" -f $intNICid)) -Name $strRegistryKeyName -Value "0"
                            
                            Write-Host -ForegroundColor Green ("   - Optimized")
						}
						
						#Xennet6 (XenServer)
						"Citrix PV Ethernet Adapter".Tolower()
						{
						
							# --- IPv4 Checksum Offload---
							$strRegistryKeyName = "*IPChecksumOffloadIPv4"
							# 0 - Disabled
							# 1 - Enabled (Transmit Only)
							# 2 - Enabled (Receive Only)
							# 3 - Enabled (Transmit and Receive) (default)
							#Set-ItemProperty -Path ("HKLM:\SYSTEM\CurrentControlSet\Control\Class\{0}\{1}" -f "{4D36E972-E325-11CE-BFC1-08002BE10318}", ( "{0:D4}" -f $intNICid)) -Name $strRegistryKeyName -Value "3"							

							# --- Large Send Offload Version 2 (IPv4) ---
							$strRegistryKeyName = "*LSOv2IPv4"
							# 0 - Disabled 
							# 1 - Enabled (default)
							Set-ItemProperty -Path ("HKLM:\SYSTEM\CurrentControlSet\Control\Class\{0}\{1}" -f "{4D36E972-E325-11CE-BFC1-08002BE10318}", ( "{0:D4}" -f $intNICid)) -Name $strRegistryKeyName -Value "0"
							
							# --- TCP Checksum Offload (IPv4) ---
							$strRegistryKeyName = "*TCPChecksumOffloadIPv4"
							# 0 - Disabled
							# 1 - Enabled (Transmit Only)
							# 2 - Enabled (Receive Only)
							# 3 - Enabled (Transmit and Receive) (default)
							#Set-ItemProperty -Path ("HKLM:\SYSTEM\CurrentControlSet\Control\Class\{0}\{1}" -f "{4D36E972-E325-11CE-BFC1-08002BE10318}", ( "{0:D4}" -f $intNICid)) -Name $strRegistryKeyName -Value "3"
							
							# --- UDP Checksum Offload (IPv4) ---
							$strRegistryKeyName = "*UDPChecksumOffloadIPv4"
							# 0 - Disabled
							# 1 - Enabled (Transmit Only)
							# 2 - Enabled (Receive Only)
							# 3 - Enabled (Transmit and Receive) (default)
							#Set-ItemProperty -Path ("HKLM:\SYSTEM\CurrentControlSet\Control\Class\{0}\{1}" -f "{4D36E972-E325-11CE-BFC1-08002BE10318}", ( "{0:D4}" -f $intNICid)) -Name $strRegistryKeyName -Value "3"

							# --- Large Receive Offload (IPv4) ---
							$strRegistryKeyName = "LROIPv4"
							# 0 - Disabled 
							# 1 - Enabled (default)
							#Set-ItemProperty -Path ("HKLM:\SYSTEM\CurrentControlSet\Control\Class\{0}\{1}" -f "{4D36E972-E325-11CE-BFC1-08002BE10318}", ( "{0:D4}" -f $intNICid)) -Name $strRegistryKeyName -Value "1"
							
							# --- "Correct TCP/UDP Checksum Value ---
							$strRegistryKeyName = "NeedChecksumValue"
							# 0 - Disabled (default)
							# 1 - Enabled 
							#Set-ItemProperty -Path ("HKLM:\SYSTEM\CurrentControlSet\Control\Class\{0}\{1}" -f "{4D36E972-E325-11CE-BFC1-08002BE10318}", ( "{0:D4}" -f $intNICid)) -Name $strRegistryKeyName -Value "0"
							
                            Write-Host -ForegroundColor Green ("   - Optimized")
						}		
						
						
						#Xennet7 (XenServer)
						"Citrix PV Network Adapter".Tolower()
						{
						
							# --- IPv4 Checksum Offload---
							$strRegistryKeyName = "*IPChecksumOffloadIPv4"
							# 0 - Disabled
							# 1 - Enabled (Transmit Only)
							# 2 - Enabled (Receive Only)
							# 3 - Enabled (Transmit and Receive) (default)
							#Set-ItemProperty -Path ("HKLM:\SYSTEM\CurrentControlSet\Control\Class\{0}\{1}" -f "{4D36E972-E325-11CE-BFC1-08002BE10318}", ( "{0:D4}" -f $intNICid)) -Name $strRegistryKeyName -Value "3"							

							# --- Large Send Offload Version 2 (IPv4) ---
							$strRegistryKeyName = "*LSOv2IPv4"
							# 0 - Disabled 
							# 1 - Enabled (default)
							Set-ItemProperty -Path ("HKLM:\SYSTEM\CurrentControlSet\Control\Class\{0}\{1}" -f "{4D36E972-E325-11CE-BFC1-08002BE10318}", ( "{0:D4}" -f $intNICid)) -Name $strRegistryKeyName -Value "0"

							# --- Large Send Offload Version 2 (IPv6) ---
							$strRegistryKeyName = "*LSOv2IPv6"
							# 0 - Disabled 
							# 1 - Enabled (default)
							Set-ItemProperty -Path ("HKLM:\SYSTEM\CurrentControlSet\Control\Class\{0}\{1}" -f "{4D36E972-E325-11CE-BFC1-08002BE10318}", ( "{0:D4}" -f $intNICid)) -Name $strRegistryKeyName -Value "0"
							
							# --- TCP Checksum Offload (IPv4) ---
							$strRegistryKeyName = "*TCPChecksumOffloadIPv4"
							# 0 - Disabled
							# 1 - Enabled (Transmit Only)
							# 2 - Enabled (Receive Only)
							# 3 - Enabled (Transmit and Receive) (default)
							#Set-ItemProperty -Path ("HKLM:\SYSTEM\CurrentControlSet\Control\Class\{0}\{1}" -f "{4D36E972-E325-11CE-BFC1-08002BE10318}", ( "{0:D4}" -f $intNICid)) -Name $strRegistryKeyName -Value "3"
							
							# --- TCP Checksum Offload (IPv6) ---
							$strRegistryKeyName = "*TCPChecksumOffloadIPv6"
							# 0 - Disabled
							# 1 - Enabled (Transmit Only)
							# 2 - Enabled (Receive Only)
							# 3 - Enabled (Transmit and Receive) (default)
							#Set-ItemProperty -Path ("HKLM:\SYSTEM\CurrentControlSet\Control\Class\{0}\{1}" -f "{4D36E972-E325-11CE-BFC1-08002BE10318}", ( "{0:D4}" -f $intNICid)) -Name $strRegistryKeyName -Value "3"
							
							# --- UDP Checksum Offload (IPv4) ---
							$strRegistryKeyName = "*UDPChecksumOffloadIPv4"
							# 0 - Disabled
							# 1 - Enabled (Transmit Only)
							# 2 - Enabled (Receive Only)
							# 3 - Enabled (Transmit and Receive) (default)
							#Set-ItemProperty -Path ("HKLM:\SYSTEM\CurrentControlSet\Control\Class\{0}\{1}" -f "{4D36E972-E325-11CE-BFC1-08002BE10318}", ( "{0:D4}" -f $intNICid)) -Name $strRegistryKeyName -Value "3"

							# --- UDP Checksum Offload (IPv6) ---
							$strRegistryKeyName = "*UDPChecksumOffloadIPv6"
							# 0 - Disabled
							# 1 - Enabled (Transmit Only)
							# 2 - Enabled (Receive Only)
							# 3 - Enabled (Transmit and Receive) (default)
							#Set-ItemProperty -Path ("HKLM:\SYSTEM\CurrentControlSet\Control\Class\{0}\{1}" -f "{4D36E972-E325-11CE-BFC1-08002BE10318}", ( "{0:D4}" -f $intNICid)) -Name $strRegistryKeyName -Value "3"

							# --- Large Receive Offload (IPv4) ---
							$strRegistryKeyName = "LROIPv4"
							# 0 - Disabled 
							# 1 - Enabled (default)
							#Set-ItemProperty -Path ("HKLM:\SYSTEM\CurrentControlSet\Control\Class\{0}\{1}" -f "{4D36E972-E325-11CE-BFC1-08002BE10318}", ( "{0:D4}" -f $intNICid)) -Name $strRegistryKeyName -Value "1"
							
							# --- Large Receive Offload (IPv6) ---
							$strRegistryKeyName = "LROIPv6"
							# 0 - Disabled 
							# 1 - Enabled (default)
							#Set-ItemProperty -Path ("HKLM:\SYSTEM\CurrentControlSet\Control\Class\{0}\{1}" -f "{4D36E972-E325-11CE-BFC1-08002BE10318}", ( "{0:D4}" -f $intNICid)) -Name $strRegistryKeyName -Value "1"
							
							# --- "Correct TCP/UDP Checksum Value ---
							$strRegistryKeyName = "NeedChecksumValue"
							# 0 - Disabled (default)
							# 1 - Enabled 
							#Set-ItemProperty -Path ("HKLM:\SYSTEM\CurrentControlSet\Control\Class\{0}\{1}" -f "{4D36E972-E325-11CE-BFC1-08002BE10318}", ( "{0:D4}" -f $intNICid)) -Name $strRegistryKeyName -Value "0"
							
                            Write-Host -ForegroundColor Green ("   - Optimized")
						}										
						
						#RTL8139a (XenServer)
						"Realtek RTL8139C+ Fast Ethernet NIC".Tolower() 
						{

							# --- Link Speed/Duplex Mode ---
							$strRegistryKeyName = "DuplexMode"
							# 1 - Auto Negotiation (default)
							# 2 -  10Mbps/Half Duplex
							# 3 -  10Mbps/Full Duplex
							# 4 - 100Mbps/Half Duplex
							# 5 - 100Mbps/Full Duplex
							#Set-ItemProperty -Path ("HKLM:\SYSTEM\CurrentControlSet\Control\Class\{0}\{1}" -f "{4D36E972-E325-11CE-BFC1-08002BE10318}", ( "{0:D4}" -f $intNICid)) -Name $strRegistryKeyName -Value "1"


							# --- Link Down Power Saving ---
							$strRegistryKeyName = "EnableLDPS"
							# 0 - Disabled (default)
							# 1 - Enabled 
							#Set-ItemProperty -Path ("HKLM:\SYSTEM\CurrentControlSet\Control\Class\{0}\{1}" -f "{4D36E972-E325-11CE-BFC1-08002BE10318}", ( "{0:D4}" -f $intNICid)) -Name $strRegistryKeyName -Value "0"


							# --- Optimal Performance ---
							$strRegistryKeyName = "OptimalPerf"
							# 0 - Disabled (default)
							# 1 - Enabled 
							#Set-ItemProperty -Path ("HKLM:\SYSTEM\CurrentControlSet\Control\Class\{0}\{1}" -f "{4D36E972-E325-11CE-BFC1-08002BE10318}", ( "{0:D4}" -f $intNICid)) -Name $strRegistryKeyName -Value "0"

							# --- Receive Buffer Size ---
							$strRegistryKeyName = "RxBufLen"
                            # 0 - 8K bytes
                            # 1 - 16K bytes
                            # 2 - 32K bytes
                            # 3 - 64K bytes (default)
							#Set-ItemProperty -Path ("HKLM:\SYSTEM\CurrentControlSet\Control\Class\{0}\{1}" -f "{4D36E972-E325-11CE-BFC1-08002BE10318}", ( "{0:D4}" -f $intNICid)) -Name $strRegistryKeyName -Value "3"
							
							Write-Host -ForegroundColor Green ("   - Optimized")
						}
						
						#vmbus (Hyper-V)
						"Microsoft Virtual Machine Bus Network Adapter".Tolower() 
						{

							#--- IPv4 Checksum Offload ---
							$strRegistryKeyName = "*IPChecksumOffloadIPv4"
							# 0 - Disabled
							# 1 - Tx Enabled
							# 2 - Rx Enabled
							# 3 - Tx and Rx Enabled (default)
							#Set-ItemProperty -Path ("HKLM:\SYSTEM\CurrentControlSet\Control\Class\{0}\{1}" -f "{4D36E972-E325-11CE-BFC1-08002BE10318}", ( "{0:D4}" -f $intNICid)) -Name $strRegistryKeyName -Value "3"
							
							#--- Jumbo Packet ---
							$strRegistryKeyName = "*JumboPacket"
							# 1514 - Disabled (Default)
							# 4088 - 4088 Bytes
							# 9014 - 9014 Bytes
							#Set-ItemProperty -Path ("HKLM:\SYSTEM\CurrentControlSet\Control\Class\{0}\{1}" -f "{4D36E972-E325-11CE-BFC1-08002BE10318}", ( "{0:D4}" -f $intNICid)) -Name $strRegistryKeyName -Value "1514"

							#--- Large Send Offload Version 2 (IPv4) ---
							$strRegistryKeyName = "*LsoV2IPv4"
							# 0 - Disabled 
							# 1 - Enabled (default)
							Set-ItemProperty -Path ("HKLM:\SYSTEM\CurrentControlSet\Control\Class\{0}\{1}" -f "{4D36E972-E325-11CE-BFC1-08002BE10318}", ( "{0:D4}" -f $intNICid)) -Name $strRegistryKeyName -Value "0"

							# --- Large Send Offload Version 2 (IPv6) ---
							$strRegistryKeyName = "*LsoV2IPv6"
							# 0 - Disabled 
							# 1 - Enabled (default)
							#Set-ItemProperty -Path ("HKLM:\SYSTEM\CurrentControlSet\Control\Class\{0}\{1}" -f "{4D36E972-E325-11CE-BFC1-08002BE10318}", ( "{0:D4}" -f $intNICid)) -Name $strRegistryKeyName -Value "1"
							
							#--- TCP Checksum Offload (IPv4) ---
							$strRegistryKeyName = "*TCPChecksumOffloadIPv4"
							# 0 - Disabled
							# 1 - Tx Enabled
							# 2 - Rx Enabled
							# 3 - Tx and Rx Enabled (default)
							#Set-ItemProperty -Path ("HKLM:\SYSTEM\CurrentControlSet\Control\Class\{0}\{1}" -f "{4D36E972-E325-11CE-BFC1-08002BE10318}", ( "{0:D4}" -f $intNICid)) -Name $strRegistryKeyName -Value "3"

							#--- TCP Checksum Offload (IPv6) ---
							$strRegistryKeyName = "*TCPChecksumOffloadIPv6"
							# 0 - Disabled
							# 1 - Tx Enabled
							# 2 - Rx Enabled
							# 3 - Tx and Rx Enabled (default)
							#Set-ItemProperty -Path ("HKLM:\SYSTEM\CurrentControlSet\Control\Class\{0}\{1}" -f "{4D36E972-E325-11CE-BFC1-08002BE10318}", ( "{0:D4}" -f $intNICid)) -Name $strRegistryKeyName -Value "3"

							#--- UDP Checksum Offload (IPv4) ---
							$strRegistryKeyName = "*UDPChecksumOffloadIPv4"
							# 0 - Disabled
							# 2 - Rx Enabled (default)
							#Set-ItemProperty -Path ("HKLM:\SYSTEM\CurrentControlSet\Control\Class\{0}\{1}" -f "{4D36E972-E325-11CE-BFC1-08002BE10318}", ( "{0:D4}" -f $intNICid)) -Name $strRegistryKeyName -Value "2"
							
							#--- UDP Checksum Offload (IPv6) ---
							$strRegistryKeyName = "*UDPChecksumOffloadIPv6"
							# 0 - Disabled
							# 2 - Rx Enabled (default)
							#Set-ItemProperty -Path ("HKLM:\SYSTEM\CurrentControlSet\Control\Class\{0}\{1}" -f "{4D36E972-E325-11CE-BFC1-08002BE10318}", ( "{0:D4}" -f $intNICid)) -Name $strRegistryKeyName -Value "2"

						
							Write-Host -ForegroundColor Green ("   - Optimized")
						}
						
						#Intel 21140  (Hyper-V legacy)
						"Intel 21140-Based PCI Fast Ethernet Adapter (Emulated)".Tolower() 
						{

							# --- Burst Length ---
							$strRegistryKeyName = "BurstLength"
							# 1 - 1 DWORD
							# 2 - 2 DWORDS
							# 4 - 4 DWORDS
							# 8 - 8 DWORDS
							# 10 - 16 DWORDS (default)
							# 20 - 32 DWORDS
							#Set-ItemProperty -Path ("HKLM:\SYSTEM\CurrentControlSet\Control\Class\{0}\{1}" -f "{4D36E972-E325-11CE-BFC1-08002BE10318}", ( "{0:D4}" -f $intNICid)) -Name $strRegistryKeyName -Value "10"

							# --- Connection Type ---
							$strRegistryKeyName = "ConnectionType"
							# 0 - AutoSense (default)
							# 8 - 100BaseTx
							# 9 - 100BaseTx Full Duplex
							# 0A - 100BaseT4
							# 0B - 100BaseFx
							# 0C - 100BaseFx Full Duplex
							# 2 - 10BaseT (Twisted Pair)
							# 3 - 10BaseT Full Duplex
							#Set-ItemProperty -Path ("HKLM:\SYSTEM\CurrentControlSet\Control\Class\{0}\{1}" -f "{4D36E972-E325-11CE-BFC1-08002BE10318}", ( "{0:D4}" -f $intNICid)) -Name $strRegistryKeyName -Value "0"

							# --- Extra Receive Buffers ---
							$strRegistryKeyName = "ExtraReceiveBuffers"
							# 0 -   0
							# 10 -  16 (default)
							# 20 -  32
							# 40 -  64
							# 80 - 128
							#Set-ItemProperty -Path ("HKLM:\SYSTEM\CurrentControlSet\Control\Class\{0}\{1}" -f "{4D36E972-E325-11CE-BFC1-08002BE10318}", ( "{0:D4}" -f $intNICid)) -Name $strRegistryKeyName -Value "10"

							# --- Extra Receive Packets ---
							$strRegistryKeyName = "ExtraReceivePackets"
							# 10 - 16
							# 20 - 32
							# 40 - 64
							# 64 - 100 (default)
							# 128 - 200
							#Set-ItemProperty -Path ("HKLM:\SYSTEM\CurrentControlSet\Control\Class\{0}\{1}" -f "{4D36E972-E325-11CE-BFC1-08002BE10318}", ( "{0:D4}" -f $intNICid)) -Name $strRegistryKeyName -Value "64"

							# ---Interrupt Mitigation ---
							$strRegistryKeyName = "InterruptMitigation"
							# 0 - Disabled (default)
							# 1 - Enabled
							#Set-ItemProperty -Path ("HKLM:\SYSTEM\CurrentControlSet\Control\Class\{0}\{1}" -f "{4D36E972-E325-11CE-BFC1-08002BE10318}", ( "{0:D4}" -f $intNICid)) -Name $strRegistryKeyName -Value "0"

							# --- Memory Read Multiple ---
							$strRegistryKeyName = "MemoryReadMultiple"
							# 0 - Disabled (default)
							# 1 - Enabled
							#Set-ItemProperty -Path ("HKLM:\SYSTEM\CurrentControlSet\Control\Class\{0}\{1}" -f "{4D36E972-E325-11CE-BFC1-08002BE10318}", ( "{0:D4}" -f $intNICid)) -Name $strRegistryKeyName -Value "0"

							# --- Network Address ---
							$strRegistryKeyName = "NetworkAddress"
							#String value
							#Set-ItemProperty -Path ("HKLM:\SYSTEM\CurrentControlSet\Control\Class\{0}\{1}" -f "{4D36E972-E325-11CE-BFC1-08002BE10318}", ( "{0:D4}" -f $intNICid)) -Name $strRegistryKeyName -Value ""

							# --- Process Transmit First ---
							$strRegistryKeyName = "ProcessTransmitFirst"
							# 0 - Disabled (default)
							# 1 - Enabled
							#Set-ItemProperty -Path ("HKLM:\SYSTEM\CurrentControlSet\Control\Class\{0}\{1}" -f "{4D36E972-E325-11CE-BFC1-08002BE10318}", ( "{0:D4}" -f $intNICid)) -Name $strRegistryKeyName -Value "0"

							# --- Receive Buffers ---
							$strRegistryKeyName = "ReceiveBuffers"
							# 8 -  8
							# 10 - 16
							# 20 - 32
							# 30 - 48 (default)
							# 40 - 64
							#Set-ItemProperty -Path ("HKLM:\SYSTEM\CurrentControlSet\Control\Class\{0}\{1}" -f "{4D36E972-E325-11CE-BFC1-08002BE10318}", ( "{0:D4}" -f $intNICid)) -Name $strRegistryKeyName -Value "30"

							# --- Store And Forward ---
							$strRegistryKeyName = "StoreAndForward"
							# 0 - Disabled (default)
							# 1 - Enabled"
							#Set-ItemProperty -Path ("HKLM:\SYSTEM\CurrentControlSet\Control\Class\{0}\{1}" -f "{4D36E972-E325-11CE-BFC1-08002BE10318}", ( "{0:D4}" -f $intNICid)) -Name $strRegistryKeyName -Value "0"

							# ---Transmit Threshold ---
							$strRegistryKeyName = "TransmitThreshold"
							# 48 -   72 bytes
							# 60 -   96 bytes (default)
							# 80 -  128 bytes
							# A0 -  160 bytes
							#Set-ItemProperty -Path ("HKLM:\SYSTEM\CurrentControlSet\Control\Class\{0}\{1}" -f "{4D36E972-E325-11CE-BFC1-08002BE10318}", ( "{0:D4}" -f $intNICid)) -Name $strRegistryKeyName -Value "60"

							# --- Transmit Threshold 100Mbps---
							$strRegistryKeyName = "TransmitThreshold100"
							# 80 -  128 bytes
							# 100 -  256 bytes
							# 200 -  512 bytes (default)
							# 400 - 1024 bytes
							#Set-ItemProperty -Path ("HKLM:\SYSTEM\CurrentControlSet\Control\Class\{0}\{1}" -f "{4D36E972-E325-11CE-BFC1-08002BE10318}", ( "{0:D4}" -f $intNICid)) -Name $strRegistryKeyName -Value "200"

							# --- Underrun Max Retries---
							$strRegistryKeyName = "UnderrunRetry"
							# 2 -  2 (default)
							# 4 -  4
							# 6 -  6
							# 8 -  8
							#Set-ItemProperty -Path ("HKLM:\SYSTEM\CurrentControlSet\Control\Class\{0}\{1}" -f "{4D36E972-E325-11CE-BFC1-08002BE10318}", ( "{0:D4}" -f $intNICid)) -Name $strRegistryKeyName -Value "2"

							# --- Underrun Threshold ---
							$strRegistryKeyName = "UnderrunThreshold"
							# 0A -  10 (default)
							# 14 -  20
							# 32 -  50
							# 64 - 100
							#Set-ItemProperty -Path ("HKLM:\SYSTEM\CurrentControlSet\Control\Class\{0}\{1}" -f "{4D36E972-E325-11CE-BFC1-08002BE10318}", ( "{0:D4}" -f $intNICid)) -Name $strRegistryKeyName -Value "0A"
							
							Write-Host -ForegroundColor Green ("   - Optimized")
						}
						
						default 
                        { 
                            Write-Host -NoNewLine " Nothing"
							Write-Host -ForegroundColor Yellow (" (contact me at info@ingmarverheij.com to add the NIC)")
                        }
					}
				}
			}
            Write-Host ""
		}
	} 
	
	#Next NIC ID
	$intNICid+=1
} while ($intNICid -lt 255)


# Request the user to reboot the machine
Write-Host -NoNewLine -ForegroundColor White "Please "
Write-Host -NoNewLine -ForegroundColor Yellow "reboot"
Write-Host -ForegroundColor White " the machine for the changes to take effect."

# Stop writing to log file
Stop-Transcript | out-null